<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TipoEstadoResource\Pages;
use App\Filament\Resources\TipoEstadoResource\RelationManagers\EstadosRelationManager;
use App\Models\TipoEstado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class TipoEstadoResource extends Resource
{
    protected static ?string $model = TipoEstado::class;

    protected static ?string $navigationIcon = 'heroicon-o-queue-list';
    
    protected static ?string $navigationLabel = 'Tipos de Estado';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 2;
    
    protected static bool $shouldRegisterNavigation = false; // Ocultar del menú

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('modulo_id')
                    ->relationship('modulo', 'nombre')
                    ->required()
                    ->label('Módulo')
                    ->searchable()
                    ->preload()
                    ->disabled()
                    ->dehydrated(),
                    
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Tipo de Estado')
                    ->live(onBlur: true)
                    ->afterStateUpdated(fn ($state, Forms\Set $set) => $set('slug', \Illuminate\Support\Str::slug($state))),
                    
                Forms\Components\TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->label('Slug (identificador único)')
                    ->unique(ignoreRecord: true)
                    ->helperText('Se genera automáticamente del nombre'),
                    
                Forms\Components\TextInput::make('icono')
                    ->maxLength(255)
                    ->label('Icono (Heroicon)')
                    ->helperText('Ej: heroicon-o-shopping-cart')
                    ->placeholder('heroicon-o-document-text'),
                    
                Forms\Components\TextInput::make('orden')
                    ->numeric()
                    ->default(0)
                    ->label('Orden')
                    ->required(),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('modulo.nombre')
                    ->label('Módulo')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('estados_count')
                    ->counts('estados')
                    ->label('Estados'),
                    
                Tables\Columns\TextColumn::make('orden')
                    ->label('Orden')
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->defaultSort('orden', 'asc')
            ->filters([
                Tables\Filters\SelectFilter::make('modulo_id')
                    ->relationship('modulo', 'nombre')
                    ->label('Módulo'),
                    
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            EstadosRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTipoEstados::route('/'),
            'create' => Pages\CreateTipoEstado::route('/create'),
            'edit' => Pages\EditTipoEstado::route('/{record}/edit'),
        ];
    }
}
