<?php

namespace App\Http\Controllers;

use App\Models\ControlGestionMensual;
use App\Models\Cliente;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class ReporteControlMensualController extends Controller
{
    public function export(Request $request)
    {
        $request->validate([
            'anios' => 'required|array',
            'periodos' => 'required|array',
            'clientes' => 'required|array',
            'formato' => 'required|in:excel,pdf',
        ]);

        // Construir query con filtros
        $query = ControlGestionMensual::with(['cliente', 'grupoEmpresa'])
            ->whereIn('periodo', $request->periodos)
            ->whereIn('cliente_id', $request->clientes);

        $registros = $query->get();

        if ($registros->isEmpty()) {
            return redirect()->back()->with('error', 'No se encontraron registros con los filtros seleccionados.');
        }

        // Generar nombre de archivo
        $timestamp = now()->format('Y-m-d_H-i-s');
        $filename = "Reporte Control Mensual - {$timestamp}";

        if ($request->formato === 'excel') {
            // Guardar mensaje en sesión ANTES de devolver el stream
            session()->flash('success', '¡Descarga exitosa! El archivo se ha generado correctamente.');
            return $this->exportarExcel($registros, $filename);
        } else {
            return $this->exportarPDF($registros, $filename);
        }
    }

    /**
     * Exportar a CSV (compatible con Excel)
     */
    private function exportarExcel($registros, $filename)
    {
        $filename .= '.csv';

        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            'Pragma' => 'no-cache',
            'Cache-Control' => 'must-revalidate, post-check=0, pre-check=0',
            'Expires' => '0'
        ];

        $callback = function() use ($registros) {
            $file = fopen('php://output', 'w');
            
            // BOM para UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));

            // Encabezados
            fputcsv($file, [
                'ID',
                'Grupo Empresa',
                'RUT Empresa',
                'Nombre Empresa',
                'Periodo',
                // Estados
                'Compras y Ventas',
                'Formulario 29',
                'Centralización Impuesto',
                'Sueldos',
                'Imposiciones Previsionales',
                'Certificados Laborales',
                'LRE',
                'Despidos Mes',
                'Registro Electrónico DT',
                'Archivo Digital',
                'Mora Presuntas',
                'Deudas Fiscales',
                'Conciliación Bancaria',
                'Balance Mensual',
                'Análisis Cuentas',
                // Formulario 29
                'Folio',
                'Monto Peso',
                'Fecha Venc Form 29',
                'Fecha Declara Plazo',
                'Fecha Pago F/P',
                'Multa Acumulada',
                'Total Pagado',
                // IVA Posterior
                'IVA Posterior',
                'Fecha Venc IVA Post',
                'Fecha Pago IVA Post',
                'Multa IVA Post',
                'HH Compra/Venta/IVA',
                // Imposiciones
                'Monto Imposiciones',
                'Fecha Venc Imposiciones',
                'Fecha Pago Imposiciones',
                'Multa Imposiciones',
                'Total Pagado Imposiciones',
                'HH Imposiciones',
                'HH Deudas',
                'HH Banco/Balance',
                // Totales HH
                'Total HH Mes',
                'HH Asignada Mes',
                'Diferencia HH Mes',
                'Valor Servicio',
                'Valor HH',
                'Fecha Creación',
                'Fecha Actualización',
            ], ';');

            // Datos
            foreach ($registros as $registro) {
                fputcsv($file, [
                    $registro->id,
                    $registro->grupoEmpresa?->nombre,
                    $registro->cliente?->rut_empresa,
                    $registro->cliente?->nombre_empresa,
                    $registro->periodo,
                    // Estados
                    $registro->comprasVentas?->nombre,
                    $registro->formulario29?->nombre,
                    $registro->centralizacionImpuesto?->nombre,
                    $registro->estSueldos?->nombre,
                    $registro->imposicionesPrevisionales?->nombre,
                    $registro->certificadosLaborales?->nombre,
                    $registro->libroElectronicoRemuneraciones?->nombre,
                    $registro->despidosMes?->nombre,
                    $registro->registroElectronicoDt?->nombre,
                    $registro->archivoDigContAnexOtros?->nombre,
                    $registro->moraPresuntas?->nombre,
                    $registro->deudasFiscales?->nombre,
                    $registro->conciliacionBancaria?->nombre,
                    $registro->balanceMensual?->nombre,
                    $registro->analisisCuentas?->nombre,
                    // Formulario 29
                    $registro->folio_cm,
                    $registro->monto_peso_cm,
                    $registro->fecha_venc_form29_cm,
                    $registro->fecha_declara_en_plazo_cm,
                    $registro->fecha_pago_f_p_cm,
                    $registro->multa_acum_clp_cm,
                    $registro->total_pagado_clp_cm,
                    // IVA Posterior
                    $registro->iva_post_clp_cm,
                    $registro->fecha_venc_iva_post_cm,
                    $registro->fecha_pago_iva_post_cm,
                    $registro->multa_iva_post_clp_cm,
                    $registro->hh_compra_venta_iva_cm,
                    // Imposiciones
                    $registro->monto_impo_cm,
                    $registro->fecha_venc_impo_cm,
                    $registro->fecha_pago_impo,
                    $registro->multa_impo_clp_cm,
                    $registro->total_pagado_clp_impo_cm,
                    $registro->hh_impo_cm,
                    $registro->hh_deuda_cm,
                    $registro->hh_banco_balance_cm,
                    // Totales HH
                    $registro->total_hh_mes_cm,
                    $registro->hh_asignada_mes,
                    $registro->diferencia_hh_mes_cm,
                    $registro->valor_servicio_cm,
                    $registro->valor_hh_cm,
                    $registro->created_at?->format('d/m/Y H:i'),
                    $registro->updated_at?->format('d/m/Y H:i'),
                ], ';');
            }

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    /**
     * Exportar a PDF (estructura preparada para futuro)
     */
    private function exportarPDF($registros, $filename)
    {
        // TODO: Implementar generación de PDF
        return redirect()->back()->with('error', 'La exportación a PDF está en desarrollo. Por favor, usa Excel por ahora.');
    }
}
