<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PerfilResource\Pages;
use App\Models\Perfil;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class PerfilResource extends Resource
{
    protected static ?string $model = Perfil::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';
    
    protected static ?string $navigationLabel = 'Perfiles';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 6;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('perfiles');
    }
    
    protected static ?string $modelLabel = 'Perfil';
    
    protected static ?string $pluralModelLabel = 'Perfiles';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Tabs::make('Tabs')
                    ->tabs([
                        // TAB 1: DATOS BÁSICOS
                        Forms\Components\Tabs\Tab::make('Datos Básicos')
                            ->icon('heroicon-o-information-circle')
                            ->schema([
                                Forms\Components\TextInput::make('nombre')
                                    ->required()
                                    ->maxLength(100)
                                    ->unique(ignoreRecord: true)
                                    ->placeholder('Ej: Gerente, Contador, Visualizador')
                                    ->columnSpanFull(),
                                    
                                Forms\Components\Textarea::make('descripcion')
                                    ->rows(3)
                                    ->placeholder('Descripción del perfil y sus responsabilidades')
                                    ->columnSpanFull(),
                                    
                                Forms\Components\Toggle::make('activo')
                                    ->label('Perfil Activo')
                                    ->default(true)
                                    ->helperText('Solo los perfiles activos pueden ser asignados a usuarios'),
                            ]),
                        
                        // TAB 2: PERMISOS DE MÓDULOS
                        Forms\Components\Tabs\Tab::make('Permisos de Módulos')
                            ->icon('heroicon-o-squares-2x2')
                            ->schema([
                                Forms\Components\Section::make('Gestión')
                                    ->schema([
                                        static::crearPermisoModulo('clientes', 'Clientes'),
                                        static::crearPermisoModulo('maestro_control_mensual', 'Maestro Control Mensual'),
                                        static::crearPermisoModulo('reportes_control_mensual', 'Reportes Control Mensual'),
                                    ]),
                                    
                                Forms\Components\Section::make('Configuración')
                                    ->schema([
                                        static::crearPermisoModulo('modulos', 'Módulos'),
                                        static::crearPermisoModulo('grupo_empresa', 'Grupo de Empresa'),
                                        static::crearPermisoModulo('software', 'Software'),
                                        static::crearPermisoModulo('bancos', 'Bancos'),
                                        static::crearPermisoModulo('estados', 'Estados'),
                                        static::crearPermisoModulo('perfiles', 'Perfiles'),
                                        static::crearPermisoModulo('usuarios', 'Usuarios'),
                                    ]),
                            ]),
                        
                        // TAB 3: PERMISOS POR CAMPO (Por ahora placeholder)
                        Forms\Components\Tabs\Tab::make('Permisos por Campo')
                            ->icon('heroicon-o-list-bullet')
                            ->schema([
                                Forms\Components\Placeholder::make('info_campos')
                                    ->content('Los permisos por campo se configurarán después de guardar el perfil y asignar los módulos con acceso.')
                                    ->columnSpanFull(),
                            ]),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    /**
     * Helper para crear campos de permisos de módulo
     */
    protected static function crearPermisoModulo(string $modulo, string $label): Forms\Components\Group
    {
        return Forms\Components\Group::make([
            Forms\Components\Checkbox::make("modulo_{$modulo}_acceso")
                ->label($label)
                ->live()
                ->columnSpan(2),
                
            Forms\Components\Checkbox::make("modulo_{$modulo}_ver")
                ->label('Ver')
                ->disabled(fn (Forms\Get $get) => !$get("modulo_{$modulo}_acceso"))
                ->columnSpan(1),
                
            Forms\Components\Checkbox::make("modulo_{$modulo}_crear")
                ->label('Crear')
                ->disabled(fn (Forms\Get $get) => !$get("modulo_{$modulo}_acceso"))
                ->columnSpan(1),
                
            Forms\Components\Checkbox::make("modulo_{$modulo}_editar")
                ->label('Editar')
                ->disabled(fn (Forms\Get $get) => !$get("modulo_{$modulo}_acceso"))
                ->columnSpan(1),
                
            Forms\Components\Checkbox::make("modulo_{$modulo}_eliminar")
                ->label('Eliminar')
                ->disabled(fn (Forms\Get $get) => !$get("modulo_{$modulo}_acceso"))
                ->columnSpan(1),
        ])->columns(6);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),
                    
                Tables\Columns\TextColumn::make('descripcion')
                    ->limit(50)
                    ->searchable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->boolean()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('usuarios_count')
                    ->label('Usuarios')
                    ->counts('usuarios')
                    ->sortable()
                    ->badge()
                    ->color('info'),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime('d/m/Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Estado')
                    ->placeholder('Todos')
                    ->trueLabel('Activos')
                    ->falseLabel('Inactivos'),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('perfiles', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('perfiles', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('perfiles', 'eliminar')),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPerfils::route('/'),
            'create' => Pages\CreatePerfil::route('/create'),
            'edit' => Pages\EditPerfil::route('/{record}/edit'),
        ];
    }
}
