<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ModuloResource\Pages;
use App\Filament\Resources\ModuloResource\RelationManagers\TiposEstadoRelationManager;
use App\Models\Modulo;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class ModuloResource extends Resource
{
    protected static ?string $model = Modulo::class;

    protected static ?string $navigationIcon = 'heroicon-o-cube';
    
    protected static ?string $navigationLabel = 'Módulos';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 1;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('modulos');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Módulo')
                    ->live(onBlur: true)
                    ->afterStateUpdated(fn ($state, Forms\Set $set) => $set('slug', \Illuminate\Support\Str::slug($state))),
                    
                Forms\Components\TextInput::make('slug')
                    ->required()
                    ->maxLength(255)
                    ->label('Slug (identificador único)')
                    ->unique(ignoreRecord: true)
                    ->helperText('Se genera automáticamente del nombre'),
                    
                Forms\Components\Textarea::make('descripcion')
                    ->maxLength(500)
                    ->label('Descripción')
                    ->rows(3),
                    
                Forms\Components\TextInput::make('orden')
                    ->numeric()
                    ->default(0)
                    ->label('Orden')
                    ->required(),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('tipos_estado_count')
                    ->counts('tiposEstado')
                    ->label('Tipos de Estado'),
                    
                Tables\Columns\TextColumn::make('orden')
                    ->label('Orden')
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->defaultSort('orden', 'asc')
            ->filters([
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('modulos', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('modulos', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('modulos', 'eliminar')),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            TiposEstadoRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListModulos::route('/'),
            'create' => Pages\CreateModulo::route('/create'),
            'edit' => Pages\EditModulo::route('/{record}/edit'),
        ];
    }
}
