<?php

namespace App\Filament\Resources;

use App\Filament\Resources\GrupoEmpresaResource\Pages;
use App\Models\GrupoEmpresa;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class GrupoEmpresaResource extends Resource
{
    protected static ?string $model = GrupoEmpresa::class;

    protected static ?string $navigationIcon = 'heroicon-o-folder-open';
    
    protected static ?string $navigationLabel = 'Grupos de Empresa';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 10;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('grupo_empresa');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Grupo'),
                    
                Forms\Components\Textarea::make('descripcion')
                    ->maxLength(500)
                    ->label('Descripción')
                    ->rows(3),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('clientes_count')
                    ->counts('clientes')
                    ->label('Clientes'),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('grupo_empresa', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('grupo_empresa', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('grupo_empresa', 'eliminar')),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListGrupoEmpresas::route('/'),
            'create' => Pages\CreateGrupoEmpresa::route('/create'),
            'edit' => Pages\EditGrupoEmpresa::route('/{record}/edit'),
        ];
    }
}
