<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ControlGestionMensualResource\Pages;
use App\Models\ControlGestionMensual;
use App\Models\Cliente;
use App\Models\GrupoEmpresa;
use App\Models\Estado;
use App\Models\TipoEstado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Validation\Rule;

class ControlGestionMensualResource extends Resource
{
    protected static ?string $model = ControlGestionMensual::class;

    protected static ?string $navigationIcon = 'heroicon-o-clipboard-document-check';
    
    protected static ?string $navigationLabel = 'Maestro Control Mensual';
    
    protected static ?string $modelLabel = 'Control Mensual';
    
    protected static ?string $pluralModelLabel = 'Controles Mensuales';
    
    protected static ?string $navigationGroup = 'Gestión';
    
    protected static ?int $navigationSort = 1;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('maestro_control_mensual');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                // SECCIÓN: CLIENTE Y PERIODO
                Forms\Components\Select::make('grupo_empresa_id')
                    ->label('Grupo de Empresa')
                    ->relationship('grupoEmpresa', 'nombre')
                    ->searchable()
                    ->preload()
                    ->live()
                    ->afterStateUpdated(function (Forms\Set $set) {
                        $set('cliente_id', null);
                    }),

                Forms\Components\Select::make('cliente_id')
                    ->label('Cliente / Empresa')
                    ->options(function (Forms\Get $get) {
                        $grupoId = $get('grupo_empresa_id');
                        
                        if ($grupoId) {
                            // Si hay grupo seleccionado, mostrar solo clientes de ese grupo
                            return Cliente::where('grupo_empresa_id', $grupoId)
                                ->where('activo', true)
                                ->pluck('nombre_empresa', 'id');
                        }
                        
                        // Si no hay grupo, mostrar todos los clientes activos
                        return Cliente::where('activo', true)
                            ->pluck('nombre_empresa', 'id');
                    })
                    ->searchable()
                    ->preload()
                    ->required()
                    ->live()
                    ->afterStateUpdated(function (Forms\Set $set, $state) {
                        if ($state) {
                            $cliente = Cliente::find($state);
                            $set('rut_empresa_display', $cliente?->rut_empresa);
                        } else {
                            $set('rut_empresa_display', null);
                        }
                    }),

                Forms\Components\TextInput::make('rut_empresa_display')
                    ->label('RUT Empresa')
                    ->disabled()
                    ->dehydrated(false)
                    ->default(function (Forms\Get $get, $record) {
                        // Al crear: obtener del cliente seleccionado
                        if (!$record && $get('cliente_id')) {
                            $cliente = Cliente::find($get('cliente_id'));
                            return $cliente?->rut_empresa;
                        }
                        // Al editar: obtener del registro existente
                        if ($record && $record->cliente_id) {
                            return $record->cliente?->rut_empresa;
                        }
                        return null;
                    })
                    ->live(),

                Forms\Components\TextInput::make('periodo')
                    ->label('Periodo (MM-YYYY)')
                    ->placeholder('01-2025')
                    ->required()
                    ->maxLength(7)
                    ->regex('/^\d{2}-\d{4}$/')
                    ->validationMessages([
                        'regex' => 'El formato debe ser MM-YYYY (ejemplo: 01-2025)',
                    ])
                    ->rules([
                        function (Forms\Get $get, $record) {
                            return Rule::unique('control_gestion_mensual', 'periodo')
                                ->where('cliente_id', $get('cliente_id'))
                                ->ignore($record?->id);
                        },
                    ])
                    ->validationMessages([
                        'unique' => 'Ya existe un registro para este cliente en el periodo seleccionado.',
                    ]),

                // SECCIÓN: ESTADOS - Compras y Ventas
                Forms\Components\Select::make('compras_ventas_id')
                    ->label('Compras y Ventas')
                    ->options(function () {
                        return self::getEstadosPorTipo('Compras y Ventas');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('formulario_29_id')
                    ->label('Formulario 29')
                    ->options(function () {
                        return self::getEstadosPorTipo('Formulario 29');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('centralizacion_impuesto_id')
                    ->label('Centralización Impuesto')
                    ->options(function () {
                        return self::getEstadosPorTipo('Centralización Impuesto');
                    })
                    ->searchable()
                    ->preload(),

                // CAMPOS: Formulario 29
                Forms\Components\TextInput::make('folio_cm')
                    ->label('Folio')
                    ->maxLength(255),

                Forms\Components\TextInput::make('monto_peso_cm')
                    ->label('Monto ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('fecha_venc_form29_cm')
                    ->label('Fecha Venc. Form 29')
                    ->maxLength(255),

                Forms\Components\TextInput::make('fecha_declara_en_plazo_cm')
                    ->label('Fecha Declara en Plazo')
                    ->maxLength(255),

                Forms\Components\TextInput::make('fecha_pago_f_p_cm')
                    ->label('Fecha Pago F/P')
                    ->maxLength(255),

                Forms\Components\TextInput::make('multa_acum_clp_cm')
                    ->label('Multa Acumulada ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('total_pagado_clp_cm')
                    ->label('Total Pagado ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                // CAMPOS: IVA Posterior
                Forms\Components\TextInput::make('iva_post_clp_cm')
                    ->label('IVA Posterior ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('fecha_venc_iva_post_cm')
                    ->label('Fecha Venc. IVA Post')
                    ->maxLength(255),

                Forms\Components\TextInput::make('fecha_pago_iva_post_cm')
                    ->label('Fecha Pago IVA Post')
                    ->maxLength(255),

                Forms\Components\TextInput::make('multa_iva_post_clp_cm')
                    ->label('Multa IVA Post ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('hh_compra_venta_iva_cm')
                    ->label('HH Compra/Venta/IVA')
                    ->maxLength(255),

                // SECCIÓN: ESTADOS - Sueldos
                Forms\Components\Select::make('est_sueldos_id')
                    ->label('Sueldos')
                    ->options(function () {
                        return self::getEstadosPorTipo('Sueldos');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('imposiciones_previsionales_id')
                    ->label('Imposiciones Previsionales')
                    ->options(function () {
                        return self::getEstadosPorTipo('Imposiciones Previsionales');
                    })
                    ->searchable()
                    ->preload(),

                // CAMPOS: Imposiciones
                Forms\Components\TextInput::make('monto_impo_cm')
                    ->label('Monto Imposiciones ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('fecha_venc_impo_cm')
                    ->label('Fecha Venc. Imposiciones')
                    ->maxLength(255),

                Forms\Components\TextInput::make('fecha_pago_impo')
                    ->label('Fecha Pago Imposiciones')
                    ->maxLength(255),

                Forms\Components\TextInput::make('multa_impo_clp_cm')
                    ->label('Multa Imposiciones ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                Forms\Components\TextInput::make('total_pagado_clp_impo_cm')
                    ->label('Total Pagado Imposiciones ($)')
                    ->numeric()
                    ->prefix('$')
                    ->default(0),

                // SECCIÓN: ESTADOS - Certificados y Registros
                Forms\Components\Select::make('certificados_laborales_id')
                    ->label('Certificados Laborales')
                    ->options(function () {
                        return self::getEstadosPorTipo('Certificados Laborales');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('libro_electronico_remuneraciones_id')
                    ->label('Libro Electrónico Remuneraciones (LRE)')
                    ->options(function () {
                        return self::getEstadosPorTipo('LRE');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('despidos_mes_id')
                    ->label('Despidos del Mes')
                    ->options(function () {
                        return self::getEstadosPorTipo('Despidos del Mes');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('registro_electronico_dt_id')
                    ->label('Registro Electrónico DT')
                    ->options(function () {
                        return self::getEstadosPorTipo('Registro Electrónico DT');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('archivo_dig_cont_anex_otros_id')
                    ->label('Archivo Digital')
                    ->options(function () {
                        return self::getEstadosPorTipo('Archivo Digital');
                    })
                    ->searchable()
                    ->preload(),

                // SECCIÓN: ESTADOS - Análisis
                Forms\Components\Select::make('mora_presuntas_id')
                    ->label('Mora Presuntas')
                    ->options(function () {
                        return self::getEstadosPorTipo('Mora Presuntas');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\TextInput::make('hh_impo_cm')
                    ->label('HH Imposiciones')
                    ->maxLength(255),

                Forms\Components\Select::make('deudas_fiscales_id')
                    ->label('Deudas Fiscales')
                    ->options(function () {
                        return self::getEstadosPorTipo('Deudas Fiscales');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\TextInput::make('hh_deuda_cm')
                    ->label('HH Deudas')
                    ->maxLength(255),

                Forms\Components\Select::make('conciliacion_bancaria_id')
                    ->label('Conciliación Bancaria')
                    ->options(function () {
                        return self::getEstadosPorTipo('Conciliación Bancaria');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('balance_mensual_id')
                    ->label('Balance Mensual')
                    ->options(function () {
                        return self::getEstadosPorTipo('Balance Mensual');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\Select::make('analisis_cuentas_id')
                    ->label('Análisis de Cuentas')
                    ->options(function () {
                        return self::getEstadosPorTipo('Análisis de Cuentas');
                    })
                    ->searchable()
                    ->preload(),

                Forms\Components\TextInput::make('hh_banco_balance_cm')
                    ->label('HH Banco/Balance')
                    ->maxLength(255),

                // CAMPOS: Totales de Horas
                Forms\Components\TextInput::make('total_hh_mes_cm')
                    ->label('Total HH Mes')
                    ->maxLength(255),

                Forms\Components\TextInput::make('hh_asignada_mes')
                    ->label('HH Asignada Mes')
                    ->maxLength(255),

                Forms\Components\TextInput::make('diferencia_hh_mes_cm')
                    ->label('Diferencia HH Mes')
                    ->maxLength(255),

                // CAMPOS: Valores
                Forms\Components\TextInput::make('valor_servicio_cm')
                    ->label('Valor Servicio')
                    ->maxLength(255),

                Forms\Components\TextInput::make('valor_hh_cm')
                    ->label('Valor HH')
                    ->maxLength(255),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('periodo')
                    ->label('Periodo')
                    ->searchable()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('cliente.nombre_empresa')
                    ->label('Empresa')
                    ->searchable()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('cliente.rut_empresa')
                    ->label('RUT')
                    ->searchable(),
                
                Tables\Columns\TextColumn::make('grupoEmpresa.nombre')
                    ->label('Grupo')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('grupo_empresa_id')
                    ->label('Grupo')
                    ->relationship('grupoEmpresa', 'nombre'),
                
                Tables\Filters\Filter::make('periodo')
                    ->form([
                        Forms\Components\TextInput::make('periodo')
                            ->label('Periodo')
                            ->placeholder('01-2025'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['periodo'],
                            fn ($query, $periodo) => $query->where('periodo', 'like', "%{$periodo}%")
                        );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('maestro_control_mensual', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('maestro_control_mensual', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('maestro_control_mensual', 'eliminar')),
                ]),
            ])
            ->defaultSort('periodo', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListControlGestionMensuals::route('/'),
            'create' => Pages\CreateControlGestionMensual::route('/create'),
            'edit' => Pages\EditControlGestionMensual::route('/{record}/edit'),
        ];
    }

    /**
     * Obtener estados filtrados por tipo
     */
    protected static function getEstadosPorTipo(string $nombreTipo): array
    {
        $tipoEstado = TipoEstado::where('nombre', $nombreTipo)->first();
        
        if (!$tipoEstado) {
            return [];
        }
        
        return Estado::where('tipo_estado_id', $tipoEstado->id)
            ->where('activo', true)
            ->pluck('nombre', 'id')
            ->toArray();
    }
}
