<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ClienteResource\Pages;
use App\Models\Cliente;
use App\Models\Estado;
use App\Models\TipoEstado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class ClienteResource extends Resource
{
    protected static ?string $model = Cliente::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-office-2';
    
    protected static ?string $navigationLabel = 'Clientes';
    
    protected static ?string $navigationGroup = 'Gestión';
    
    protected static ?int $navigationSort = 1;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('clientes');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Tabs::make('Información del Cliente')
                    ->tabs([
                        // TAB 1: Información Básica
                        Forms\Components\Tabs\Tab::make('Información Básica')
                            ->schema([
                                Forms\Components\Select::make('grupo_empresa_id')
                                    ->relationship('grupoEmpresa', 'nombre', fn ($query) => $query->where('activo', true))
                                    ->searchable()
                                    ->preload()
                                    ->label('Grupo de Empresa')
                                    ->createOptionForm([
                                        Forms\Components\TextInput::make('nombre')
                                            ->required()
                                            ->label('Nombre del Grupo'),
                                        Forms\Components\Textarea::make('descripcion')
                                            ->label('Descripción'),
                                    ]),
                                    
                                Forms\Components\TextInput::make('nombre_empresa')
                                    ->required()
                                    ->maxLength(255)
                                    ->label('Nombre de la Empresa'),
                                    
                                Forms\Components\TextInput::make('rut_empresa')
                                    ->required()
                                    ->unique(ignoreRecord: true)
                                    ->maxLength(255)
                                    ->label('RUT de la Empresa')
                                    ->placeholder('12.345.678-9'),
                                    
                                Forms\Components\TextInput::make('nombre_rep')
                                    ->maxLength(255)
                                    ->label('Nombre del Representante'),
                                    
                                Forms\Components\TextInput::make('rut_rep')
                                    ->maxLength(255)
                                    ->label('RUT del Representante')
                                    ->placeholder('12.345.678-9'),
                                    
                                Forms\Components\Textarea::make('direccion_emp')
                                    ->maxLength(500)
                                    ->label('Dirección de la Empresa')
                                    ->rows(3),
                                    
                                Forms\Components\Toggle::make('activo')
                                    ->default(true)
                                    ->label('Cliente Activo'),
                                
                                // NUEVOS CAMPOS - SOCIOS
                                Forms\Components\Section::make('Información de Socios')
                                    ->schema([
                                        Forms\Components\Fieldset::make('Socio 1')
                                            ->schema([
                                                Forms\Components\TextInput::make('nombre_socio_1')
                                                    ->maxLength(255)
                                                    ->label('Nombre Socio 1'),
                                                    
                                                Forms\Components\TextInput::make('rut_socio_1')
                                                    ->maxLength(255)
                                                    ->label('RUT Socio 1')
                                                    ->placeholder('12.345.678-9'),
                                            ])
                                            ->columns(2),
                                            
                                        Forms\Components\Fieldset::make('Socio 2')
                                            ->schema([
                                                Forms\Components\TextInput::make('nombre_socio_2')
                                                    ->maxLength(255)
                                                    ->label('Nombre Socio 2'),
                                                    
                                                Forms\Components\TextInput::make('rut_socio_2')
                                                    ->maxLength(255)
                                                    ->label('RUT Socio 2')
                                                    ->placeholder('12.345.678-9'),
                                            ])
                                            ->columns(2),
                                            
                                        Forms\Components\Fieldset::make('Socio 3')
                                            ->schema([
                                                Forms\Components\TextInput::make('nombre_socio_3')
                                                    ->maxLength(255)
                                                    ->label('Nombre Socio 3'),
                                                    
                                                Forms\Components\TextInput::make('rut_socio_3')
                                                    ->maxLength(255)
                                                    ->label('RUT Socio 3')
                                                    ->placeholder('12.345.678-9'),
                                            ])
                                            ->columns(2),
                                    ])
                                    ->columnSpanFull()
                                    ->collapsible(),
                            ])
                            ->columns(2),
                            
                        // TAB 2: Software Principal
                        Forms\Components\Tabs\Tab::make('Software Principal')
                            ->schema([
                                Forms\Components\Select::make('software_id')
                                    ->relationship('software', 'nombre', fn ($query) => $query->where('activo', true))
                                    ->searchable()
                                    ->preload()
                                    ->label('Software Principal')
                                    ->createOptionForm([
                                        Forms\Components\TextInput::make('nombre')
                                            ->required()
                                            ->label('Nombre del Software'),
                                        Forms\Components\Textarea::make('descripcion')
                                            ->label('Descripción'),
                                    ]),
                                    
                                Forms\Components\TextInput::make('nom_en_sw')
                                    ->maxLength(255)
                                    ->label('Nombre en el Software'),
                                    
                                Forms\Components\TextInput::make('usuario_sw_main')
                                    ->maxLength(255)
                                    ->label('Usuario'),
                                    
                                Forms\Components\TextInput::make('clave_sw_main')
                                    ->password()
                                    ->revealable()
                                    ->maxLength(255)
                                    ->label('Contraseña')
                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                            ])
                            ->columns(2),
                            
                        // TAB 3: Accesos SII
                        Forms\Components\Tabs\Tab::make('Accesos SII')
                            ->schema([
                                Forms\Components\Fieldset::make('Acceso Personal')
                                    ->schema([
                                        Forms\Components\TextInput::make('rut_sii')
                                            ->maxLength(255)
                                            ->label('RUT SII')
                                            ->placeholder('12.345.678-9'),
                                            
                                        Forms\Components\TextInput::make('clave_sii')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Clave SII')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                                    
                                Forms\Components\Fieldset::make('Acceso Empresa')
                                    ->schema([
                                        Forms\Components\TextInput::make('rut_sii_empresa')
                                            ->maxLength(255)
                                            ->label('RUT SII Empresa')
                                            ->placeholder('12.345.678-9'),
                                            
                                        Forms\Components\TextInput::make('clave_sii_empresa')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Clave SII Empresa')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                                    
                                Forms\Components\TextInput::make('clave_unica')
                                    ->password()
                                    ->revealable()
                                    ->maxLength(255)
                                    ->label('Clave Única')
                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null)
                                    ->columnSpanFull(),
                                
                                // NUEVOS CAMPOS - INFORMACIÓN DE SOCIOS SII
                                Forms\Components\Section::make('Información de Socios')
                                    ->schema([
                                        Forms\Components\Fieldset::make('Socio 1')
                                            ->schema([
                                                Forms\Components\TextInput::make('rut_sii_socio_1')
                                                    ->maxLength(255)
                                                    ->label('RUT SII Socio 1')
                                                    ->placeholder('12.345.678-9'),
                                                    
                                                Forms\Components\TextInput::make('clave_sii_socio_1')
                                                    ->password()
                                                    ->revealable()
                                                    ->maxLength(255)
                                                    ->label('Clave SII Socio 1')
                                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                            ])
                                            ->columns(2),
                                            
                                        Forms\Components\Fieldset::make('Socio 2')
                                            ->schema([
                                                Forms\Components\TextInput::make('rut_sii_socio_2')
                                                    ->maxLength(255)
                                                    ->label('RUT SII Socio 2')
                                                    ->placeholder('12.345.678-9'),
                                                    
                                                Forms\Components\TextInput::make('clave_sii_socio_2')
                                                    ->password()
                                                    ->revealable()
                                                    ->maxLength(255)
                                                    ->label('Clave SII Socio 2')
                                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                            ])
                                            ->columns(2),
                                            
                                        Forms\Components\Fieldset::make('Socio 3')
                                            ->schema([
                                                Forms\Components\TextInput::make('rut_sii_socio_3')
                                                    ->maxLength(255)
                                                    ->label('RUT SII Socio 3')
                                                    ->placeholder('12.345.678-9'),
                                                    
                                                Forms\Components\TextInput::make('clave_sii_socio_3')
                                                    ->password()
                                                    ->revealable()
                                                    ->maxLength(255)
                                                    ->label('Clave SII Socio 3')
                                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                            ])
                                            ->columns(2),
                                    ])
                                    ->columnSpanFull()
                                    ->collapsible(),
                                
                                // NUEVOS CAMPOS - INFORMACIÓN TRIBUTARIA
                                Forms\Components\Section::make('Información Tributaria')
                                    ->schema([
                                        Forms\Components\DatePicker::make('fecha_inicio_actividades')
                                            ->label('Fecha Inicio de Actividades')
                                            ->displayFormat('d/m/Y')
                                            ->native(false),
                                            
                                        Forms\Components\Select::make('anio_tributario')
                                            ->label('Año Tributario')
                                            ->options([
                                                2026 => '2026',
                                                2027 => '2027',
                                                2028 => '2028',
                                                2029 => '2029',
                                                2030 => '2030',
                                            ])
                                            ->searchable(),
                                            
                                        Forms\Components\Select::make('regimen_tributario_id')
                                            ->label('Régimen Tributario')
                                            ->options(function () {
                                                return self::getEstadosPorTipo('Régimen Tributario');
                                            })
                                            ->searchable()
                                            ->preload(),
                                    ])
                                    ->columns(3)
                                    ->columnSpanFull()
                                    ->collapsible(),
                            ])
                            ->columns(2),
                            
                        // TAB 4: Previred y DT
                        Forms\Components\Tabs\Tab::make('Previred y DT')
                            ->schema([
                                Forms\Components\TextInput::make('usuario_previred')
                                    ->maxLength(255)
                                    ->label('Usuario Previred'),
                                    
                                Forms\Components\TextInput::make('clave_previred')
                                    ->password()
                                    ->revealable()
                                    ->maxLength(255)
                                    ->label('Clave Previred')
                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    
                                Forms\Components\TextInput::make('clave_dt')
                                    ->password()
                                    ->revealable()
                                    ->maxLength(255)
                                    ->label('Clave DT (Dirección del Trabajo)')
                                    ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                            ])
                            ->columns(2),
                            
                        // TAB 5: Otros Softwares
                        Forms\Components\Tabs\Tab::make('Otros Softwares')
                            ->schema([
                                Forms\Components\Fieldset::make('Defontana')
                                    ->schema([
                                        Forms\Components\TextInput::make('usuario_defontana')
                                            ->maxLength(255)
                                            ->label('Usuario'),
                                            
                                        Forms\Components\TextInput::make('clave_defontana')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Contraseña')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                                    
                                Forms\Components\Fieldset::make('Nubox')
                                    ->schema([
                                        Forms\Components\TextInput::make('usuario_nubox')
                                            ->maxLength(255)
                                            ->label('Usuario'),
                                            
                                        Forms\Components\TextInput::make('clave_nubox')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Contraseña')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                            ]),
                            
                        // TAB 6: Información Bancaria
                        Forms\Components\Tabs\Tab::make('Información Bancaria')
                            ->schema([
                                Forms\Components\Fieldset::make('Banco 1')
                                    ->schema([
                                        Forms\Components\Select::make('banco_id')
                                            ->relationship('banco', 'nombre', fn ($query) => $query->where('activo', true))
                                            ->searchable()
                                            ->preload()
                                            ->label('Banco')
                                            ->createOptionForm([
                                                Forms\Components\TextInput::make('nombre')
                                                    ->required()
                                                    ->label('Nombre del Banco'),
                                                Forms\Components\TextInput::make('codigo')
                                                    ->label('Código'),
                                            ]),
                                            
                                        Forms\Components\TextInput::make('numero_cuenta')
                                            ->maxLength(255)
                                            ->label('Número de Cuenta'),
                                            
                                        Forms\Components\Select::make('tipo_ingreso_banco')
                                            ->options([
                                                'web' => 'Web',
                                                'api' => 'API',
                                                'archivo' => 'Archivo',
                                                'manual' => 'Manual',
                                            ])
                                            ->label('Tipo de Ingreso'),
                                            
                                        Forms\Components\TextInput::make('usuario_banco')
                                            ->maxLength(255)
                                            ->label('Usuario'),
                                            
                                        Forms\Components\TextInput::make('clave_us_banco')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Contraseña')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                                    
                                Forms\Components\Fieldset::make('Banco 2')
                                    ->schema([
                                        Forms\Components\Select::make('banco_id_2')
                                            ->relationship('banco2', 'nombre', fn ($query) => $query->where('activo', true))
                                            ->searchable()
                                            ->preload()
                                            ->label('Banco')
                                            ->createOptionForm([
                                                Forms\Components\TextInput::make('nombre')
                                                    ->required()
                                                    ->label('Nombre del Banco'),
                                                Forms\Components\TextInput::make('codigo')
                                                    ->label('Código'),
                                            ]),
                                            
                                        Forms\Components\TextInput::make('numero_cuenta_2')
                                            ->maxLength(255)
                                            ->label('Número de Cuenta'),
                                            
                                        Forms\Components\Select::make('tipo_ingreso_banco_2')
                                            ->options([
                                                'web' => 'Web',
                                                'api' => 'API',
                                                'archivo' => 'Archivo',
                                                'manual' => 'Manual',
                                            ])
                                            ->label('Tipo de Ingreso'),
                                            
                                        Forms\Components\TextInput::make('usuario_banco_2')
                                            ->maxLength(255)
                                            ->label('Usuario'),
                                            
                                        Forms\Components\TextInput::make('clave_us_banco_2')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Contraseña')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                                    
                                Forms\Components\Fieldset::make('Banco 3')
                                    ->schema([
                                        Forms\Components\Select::make('banco_id_3')
                                            ->relationship('banco3', 'nombre', fn ($query) => $query->where('activo', true))
                                            ->searchable()
                                            ->preload()
                                            ->label('Banco')
                                            ->createOptionForm([
                                                Forms\Components\TextInput::make('nombre')
                                                    ->required()
                                                    ->label('Nombre del Banco'),
                                                Forms\Components\TextInput::make('codigo')
                                                    ->label('Código'),
                                            ]),
                                            
                                        Forms\Components\TextInput::make('numero_cuenta_3')
                                            ->maxLength(255)
                                            ->label('Número de Cuenta'),
                                            
                                        Forms\Components\Select::make('tipo_ingreso_banco_3')
                                            ->options([
                                                'web' => 'Web',
                                                'api' => 'API',
                                                'archivo' => 'Archivo',
                                                'manual' => 'Manual',
                                            ])
                                            ->label('Tipo de Ingreso'),
                                            
                                        Forms\Components\TextInput::make('usuario_banco_3')
                                            ->maxLength(255)
                                            ->label('Usuario'),
                                            
                                        Forms\Components\TextInput::make('clave_us_banco_3')
                                            ->password()
                                            ->revealable()
                                            ->maxLength(255)
                                            ->label('Contraseña')
                                            ->dehydrateStateUsing(fn ($state) => $state ? $state : null),
                                    ])
                                    ->columns(2),
                            ]),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('rut_empresa')
                    ->label('RUT')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('nombre_empresa')
                    ->label('Empresa')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('grupoEmpresa.nombre')
                    ->label('Grupo')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),
                    
                Tables\Columns\TextColumn::make('software.nombre')
                    ->label('Software')
                    ->searchable()
                    ->toggleable(),
                    
                Tables\Columns\TextColumn::make('banco.nombre')
                    ->label('Banco')
                    ->searchable()
                    ->toggleable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('grupo_empresa_id')
                    ->relationship('grupoEmpresa', 'nombre')
                    ->label('Grupo'),
                    
                Tables\Filters\SelectFilter::make('software_id')
                    ->relationship('software', 'nombre')
                    ->label('Software'),
                    
                Tables\Filters\SelectFilter::make('banco_id')
                    ->relationship('banco', 'nombre')
                    ->label('Banco'),
                    
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('clientes', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('clientes', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('clientes', 'eliminar')),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListClientes::route('/'),
            'create' => Pages\CreateCliente::route('/create'),
            'edit' => Pages\EditCliente::route('/{record}/edit'),
        ];
    }
    
    /**
     * Obtener estados filtrados por tipo
     */
    protected static function getEstadosPorTipo(string $nombreTipo): array
    {
        $tipoEstado = TipoEstado::where('nombre', $nombreTipo)->first();
        
        if (!$tipoEstado) {
            return [];
        }
        
        return Estado::where('tipo_estado_id', $tipoEstado->id)
            ->where('activo', true)
            ->pluck('nombre', 'id')
            ->toArray();
    }
}
