<?php

namespace App\Filament\Resources;

use App\Filament\Resources\BancoResource\Pages;
use App\Models\Banco;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class BancoResource extends Resource
{
    protected static ?string $model = Banco::class;

    protected static ?string $navigationIcon = 'heroicon-o-building-library';
    
    protected static ?string $navigationLabel = 'Bancos';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 12;
    
    public static function shouldRegisterNavigation(): bool
    {
        $user = auth()->user();
        
        if (!$user || !$user->perfil) {
            return false;
        }
        
        // Super Admin siempre tiene acceso
        if ($user->perfil->nombre === 'Super Admin' || $user->perfil->nombre === 'Administrador') {
            return true;
        }
        
        return $user->tieneAccesoModulo('bancos');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Banco'),
                    
                Forms\Components\TextInput::make('codigo')
                    ->maxLength(255)
                    ->label('Código')
                    ->helperText('Código identificador del banco'),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('codigo')
                    ->label('Código')
                    ->searchable(),
                    
                Tables\Columns\TextColumn::make('clientes_count')
                    ->counts('clientes')
                    ->label('Clientes'),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('bancos', 'editar')),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn () => auth()->user()?->puedeEnModulo('bancos', 'eliminar')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn () => auth()->user()?->puedeEnModulo('bancos', 'eliminar')),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBancos::route('/'),
            'create' => Pages\CreateBanco::route('/create'),
            'edit' => Pages\EditBanco::route('/{record}/edit'),
        ];
    }
}
