<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Perfil extends Model
{
    protected $table = 'perfiles';
    
    protected $fillable = [
        'nombre',
        'descripcion',
        'activo',
    ];

    protected $casts = [
        'activo' => 'boolean',
    ];

    /**
     * Relación con usuarios
     */
    public function usuarios(): HasMany
    {
        return $this->hasMany(User::class, 'perfil_id');
    }

    /**
     * Relación con permisos de módulos
     */
    public function permisoModulos(): HasMany
    {
        return $this->hasMany(PerfilModulo::class);
    }

    /**
     * Relación con permisos de campos
     */
    public function permisoCampos(): HasMany
    {
        return $this->hasMany(PerfilCampoPermiso::class);
    }
    
    /**
     * Verificar si tiene acceso a un módulo
     */
    public function tieneAccesoModulo(string $modulo): bool
    {
        return $this->permisoModulos()
            ->where('modulo', $modulo)
            ->where('tiene_acceso', true)
            ->exists();
    }
    
    /**
     * Verificar si puede realizar acción en módulo
     */
    public function puedeEnModulo(string $modulo, string $accion): bool
    {
        $permiso = $this->permisoModulos()
            ->where('modulo', $modulo)
            ->first();
            
        if (!$permiso) {
            return false;
        }
        
        return match($accion) {
            'ver' => $permiso->puede_ver,
            'crear' => $permiso->puede_crear,
            'editar' => $permiso->puede_editar,
            'eliminar' => $permiso->puede_eliminar,
            default => false,
        };
    }
}
