<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use App\Models\Perfil;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Hash;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';
    
    protected static ?string $navigationLabel = 'Usuarios';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 7;
    
    protected static ?string $modelLabel = 'Usuario';
    
    protected static ?string $pluralModelLabel = 'Usuarios';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Información Personal')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('Nombre Completo')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('Ej: Juan Pérez González'),
                            
                        Forms\Components\TextInput::make('email')
                            ->label('Correo Electrónico')
                            ->email()
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(255)
                            ->placeholder('usuario@empresa.cl'),
                            
                        Forms\Components\TextInput::make('cargo')
                            ->label('Cargo')
                            ->maxLength(100)
                            ->placeholder('Ej: Contador Senior, Gerente de Operaciones'),
                            
                        Forms\Components\TextInput::make('telefono')
                            ->label('Teléfono')
                            ->tel()
                            ->maxLength(20)
                            ->placeholder('Ej: +56 9 1234 5678'),
                    ])
                    ->columns(2),
                
                Forms\Components\Section::make('Acceso y Seguridad')
                    ->schema([
                        Forms\Components\Select::make('perfil_id')
                            ->label('Perfil')
                            ->relationship('perfil', 'nombre')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live()
                            ->helperText('El perfil determina los permisos del usuario'),
                            
                        Forms\Components\TextInput::make('password')
                            ->label('Contraseña')
                            ->password()
                            ->revealable()
                            ->dehydrateStateUsing(fn ($state) => Hash::make($state))
                            ->dehydrated(fn ($state) => filled($state))
                            ->required(fn (string $context): bool => $context === 'create')
                            ->maxLength(255)
                            ->placeholder('Mínimo 8 caracteres')
                            ->helperText(fn (string $context): string => 
                                $context === 'edit' 
                                    ? 'Dejar en blanco para mantener la contraseña actual' 
                                    : 'La contraseña debe tener al menos 8 caracteres'
                            ),
                            
                        Forms\Components\Toggle::make('activo')
                            ->label('Usuario Activo')
                            ->default(true)
                            ->helperText('Solo los usuarios activos pueden iniciar sesión'),
                    ])
                    ->columns(2),
                
                Forms\Components\Section::make('Resumen de Permisos')
                    ->schema([
                        Forms\Components\Placeholder::make('permisos_info')
                            ->label('')
                            ->content(function (Forms\Get $get) {
                                $perfilId = $get('perfil_id');
                                
                                if (!$perfilId) {
                                    return 'Selecciona un perfil para ver los permisos asignados.';
                                }
                                
                                $perfil = Perfil::with('permisoModulos')->find($perfilId);
                                
                                if (!$perfil) {
                                    return 'Perfil no encontrado.';
                                }
                                
                                $permisos = $perfil->permisoModulos()
                                    ->where('tiene_acceso', true)
                                    ->get();
                                
                                if ($permisos->isEmpty()) {
                                    return 'Este perfil no tiene permisos asignados aún.';
                                }
                                
                                $texto = "**Módulos con acceso:**\n\n";
                                
                                foreach ($permisos as $permiso) {
                                    $acciones = [];
                                    if ($permiso->puede_ver) $acciones[] = 'Ver';
                                    if ($permiso->puede_crear) $acciones[] = 'Crear';
                                    if ($permiso->puede_editar) $acciones[] = 'Editar';
                                    if ($permiso->puede_eliminar) $acciones[] = 'Eliminar';
                                    
                                    $moduloLabel = ucfirst(str_replace('_', ' ', $permiso->modulo));
                                    $texto .= "• **{$moduloLabel}:** " . implode(', ', $acciones) . "\n";
                                }
                                
                                return new \Illuminate\Support\HtmlString(
                                    str($texto)->markdown()->toHtmlString()
                                );
                            })
                            ->columnSpanFull(),
                    ])
                    ->collapsible()
                    ->collapsed(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),
                    
                Tables\Columns\TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->icon('heroicon-m-envelope'),
                    
                Tables\Columns\TextColumn::make('perfil.nombre')
                    ->label('Perfil')
                    ->badge()
                    ->searchable()
                    ->sortable()
                    ->color(fn (string $state): string => match ($state) {
                        'Administrador', 'Super Admin' => 'danger',
                        'Gerente' => 'warning',
                        'Analista' => 'info',
                        'Reportes' => 'success',
                        default => 'gray',
                    }),
                    
                Tables\Columns\TextColumn::make('cargo')
                    ->label('Cargo')
                    ->searchable()
                    ->toggleable(),
                    
                Tables\Columns\TextColumn::make('telefono')
                    ->label('Teléfono')
                    ->toggleable()
                    ->icon('heroicon-m-phone'),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('ultimo_acceso')
                    ->label('Último Acceso')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime('d/m/Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('perfil_id')
                    ->label('Perfil')
                    ->relationship('perfil', 'nombre')
                    ->multiple()
                    ->preload(),
                    
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Estado')
                    ->placeholder('Todos')
                    ->trueLabel('Activos')
                    ->falseLabel('Inactivos'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}
