<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Modulo;
use App\Models\TipoEstado;
use App\Models\Estado;
use Illuminate\Support\Str;

class MigrarEstadosANuevaEstructura extends Command
{
    protected $signature = 'estados:migrar';
    protected $description = 'Migrar estados antiguos a la nueva estructura Modulos > TiposEstado > Estados';

    public function handle()
    {
        $this->info('Iniciando migración de estados...');
        
        // Mapeo de tipos a nombres legibles
        $tiposMap = [
            'compras_ventas' => ['nombre' => 'Compras y Ventas', 'icono' => 'heroicon-o-shopping-cart', 'orden' => 1],
            'formulario_29' => ['nombre' => 'Formulario 29', 'icono' => 'heroicon-o-document-text', 'orden' => 2],
            'centralizacion_impuesto' => ['nombre' => 'Centralización Impuesto', 'icono' => 'heroicon-o-document-chart-bar', 'orden' => 3],
            'deudas_fiscales' => ['nombre' => 'Deudas Fiscales', 'icono' => 'heroicon-o-exclamation-triangle', 'orden' => 4],
            'conciliacion_bancaria' => ['nombre' => 'Conciliación Bancaria', 'icono' => 'heroicon-o-banknotes', 'orden' => 5],
            'balance_mensual' => ['nombre' => 'Balance Mensual', 'icono' => 'heroicon-o-scale', 'orden' => 6],
            'analisis_cuentas' => ['nombre' => 'Análisis de Cuentas', 'icono' => 'heroicon-o-chart-bar', 'orden' => 7],
            'mora_presuntas' => ['nombre' => 'Mora Presuntas', 'icono' => 'heroicon-o-clock', 'orden' => 8],
            'est_sueldos' => ['nombre' => 'Sueldos', 'icono' => 'heroicon-o-currency-dollar', 'orden' => 9],
            'certificados_laborales' => ['nombre' => 'Certificados Laborales', 'icono' => 'heroicon-o-document-check', 'orden' => 10],
            'imposiciones_previsionales' => ['nombre' => 'Imposiciones Previsionales', 'icono' => 'heroicon-o-building-library', 'orden' => 11],
            'libro_electronico_remuneraciones' => ['nombre' => 'LRE', 'icono' => 'heroicon-o-book-open', 'orden' => 12],
            'despidos_mes' => ['nombre' => 'Despidos del Mes', 'icono' => 'heroicon-o-user-minus', 'orden' => 13],
            'registro_electronico_dt' => ['nombre' => 'Registro Electrónico DT', 'icono' => 'heroicon-o-clipboard-document-list', 'orden' => 14],
            'archivo_dig_cont_anex_otros' => ['nombre' => 'Archivo Digital', 'icono' => 'heroicon-o-archive-box', 'orden' => 15],
            'compra_ventas_new' => ['nombre' => 'Compras y Ventas New', 'icono' => 'heroicon-o-shopping-cart', 'orden' => 16],
        ];
        
        // Paso 1: Crear el módulo único
        $this->info('Paso 1: Creando módulo...');
        $modulo = Modulo::firstOrCreate(
            ['slug' => 'estados-control-mensual'],
            [
                'nombre' => 'Estados Control Mensual',
                'descripcion' => 'Módulo principal para control mensual de estados',
                'orden' => 1,
                'activo' => true,
            ]
        );
        $this->info("  ✓ Módulo: Estados Control Mensual");
        
        // Paso 2: Crear tipos de estado
        $this->info('Paso 2: Creando tipos de estado...');
        $tiposEstado = [];
        
        foreach ($tiposMap as $tipoSlug => $config) {
            $tipoEstado = TipoEstado::firstOrCreate(
                ['slug' => $tipoSlug],
                [
                    'modulo_id' => $modulo->id,
                    'nombre' => $config['nombre'],
                    'icono' => $config['icono'],
                    'orden' => $config['orden'],
                    'activo' => true,
                ]
            );
            
            $tiposEstado[$tipoSlug] = $tipoEstado;
            $this->info("  ✓ Tipo Estado: {$config['nombre']}");
        }
        
        // Paso 3: Actualizar estados existentes
        $this->info('Paso 3: Vinculando estados existentes...');
        
        $estadosActualizados = 0;
        
        foreach ($tiposMap as $tipoSlug => $config) {
            $tipoEstado = $tiposEstado[$tipoSlug];
            
            $estados = Estado::where('tipo', $tipoSlug)
                ->whereNull('tipo_estado_id')
                ->get();
            
            foreach ($estados as $estado) {
                $estado->tipo_estado_id = $tipoEstado->id;
                $estado->save();
                $estadosActualizados++;
            }
            
            if ($estados->count() > 0) {
                $this->info("  ✓ Vinculados " . $estados->count() . " estados a '{$config['nombre']}'");
            }
        }
        
        $this->info('');
        $this->info("✅ Migración completada!");
        $this->info("   - Módulo creado: Estados Control Mensual");
        $this->info("   - Tipos de estado creados: " . count($tiposEstado));
        $this->info("   - Estados vinculados: " . $estadosActualizados);
        
        return 0;
    }
}
