<?php

namespace App\Filament\Resources;

use App\Models\Estado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class EstadoNuevoResource extends Resource
{
    protected static ?string $model = Estado::class;

    protected static ?string $navigationIcon = 'heroicon-o-tag';
    
    protected static ?string $navigationLabel = 'Estados';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 3;
    
    protected static bool $shouldRegisterNavigation = false; // Ocultar del menú

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('tipo_estado_id')
                    ->relationship('tipoEstado', 'nombre', function ($query) {
                        $query->with('modulo')
                            ->orderBy('nombre');
                    })
                    ->getOptionLabelFromRecordUsing(fn ($record) => $record->modulo->nombre . ' > ' . $record->nombre)
                    ->required()
                    ->label('Tipo de Estado')
                    ->searchable()
                    ->preload(),
                    
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Estado')
                    ->helperText('Ej: Ingresado, No Ingresado, Revisado, etc.'),
                    
                Forms\Components\Select::make('color')
                    ->options([
                        'success' => 'Verde (Éxito)',
                        'danger' => 'Rojo (Peligro)',
                        'warning' => 'Amarillo (Advertencia)',
                        'info' => 'Azul (Información)',
                        'gray' => 'Gris (Neutral)',
                    ])
                    ->label('Color')
                    ->default('gray')
                    ->required(),
                    
                Forms\Components\TextInput::make('orden')
                    ->numeric()
                    ->default(0)
                    ->label('Orden')
                    ->required(),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('tipoEstado.modulo.nombre')
                    ->label('Módulo')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('tipoEstado.nombre')
                    ->label('Tipo de Estado')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Estado')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('color')
                    ->label('Color')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'success' => 'Verde',
                        'danger' => 'Rojo',
                        'warning' => 'Amarillo',
                        'info' => 'Azul',
                        'gray' => 'Gris',
                        default => $state,
                    })
                    ->color(fn (string $state): string => $state),
                    
                Tables\Columns\TextColumn::make('orden')
                    ->label('Orden')
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->defaultSort('orden', 'asc')
            ->filters([
                Tables\Filters\SelectFilter::make('tipo_estado_id')
                    ->relationship('tipoEstado', 'nombre')
                    ->label('Tipo de Estado')
                    ->searchable()
                    ->preload(),
                    
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
