<?php
namespace App\Filament\Resources\Estados;

use App\Models\Estado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

abstract class BaseEstadoPorTipoResource extends Resource
{
    protected static ?string $model = Estado::class;
    
    protected static ?string $navigationGroup = 'Estados';
    
    abstract protected static function tipo(): string;
    
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->where('tipo', static::tipo())
            ->orderBy('orden');
    }
    
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Hidden::make('tipo')
                    ->default(static::tipo()),
                    
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Estado'),
                    
                Forms\Components\Select::make('color')
                    ->options([
                        'success' => 'Verde (Éxito)',
                        'danger' => 'Rojo (Peligro)',
                        'warning' => 'Amarillo (Advertencia)',
                        'info' => 'Azul (Información)',
                        'gray' => 'Gris (Neutral)',
                    ])
                    ->label('Color')
                    ->default('gray')
                    ->required(),
                    
                Forms\Components\TextInput::make('orden')
                    ->numeric()
                    ->default(0)
                    ->label('Orden')
                    ->required()
                    ->helperText('Número que define el orden de aparición'),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo')
                    ->helperText('Desactiva este estado si no quieres que aparezca en el sistema'),
            ]);
    }
    
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('color')
                    ->label('Color')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'success' => 'Verde',
                        'danger' => 'Rojo',
                        'warning' => 'Amarillo',
                        'info' => 'Azul',
                        'gray' => 'Gris',
                        default => $state,
                    })
                    ->color(fn (string $state): string => $state),
                    
                Tables\Columns\TextColumn::make('orden')
                    ->label('Orden')
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
                    
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Creado')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->defaultSort('orden', 'asc')
            ->actions([
                Tables\Actions\EditAction::make(),
            ]);
    }
    
    public static function canCreate(): bool
    {
        return false;
    }
    
    public static function canDeleteAny(): bool
    {
        return false;
    }
}
