<?php

namespace App\Filament\Resources;

use App\Filament\Resources\EstadoPadreResource\Pages;
use App\Models\Estado;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class EstadoPadreResource extends Resource
{
    protected static ?string $model = Estado::class;

    protected static ?string $navigationIcon = 'heroicon-o-folder';
    
    protected static ?string $navigationLabel = 'Estados Padre';
    
    protected static ?string $navigationGroup = 'Configuración';
    
    protected static ?int $navigationSort = 1;

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->padres()->orderBy('orden');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('tipo')
                    ->required()
                    ->maxLength(255)
                    ->label('Tipo (identificador único)')
                    ->helperText('Ej: compras_ventas, formulario_29, etc.')
                    ->unique(ignoreRecord: true),
                    
                Forms\Components\TextInput::make('nombre')
                    ->required()
                    ->maxLength(255)
                    ->label('Nombre del Estado Padre'),
                    
                Forms\Components\Select::make('color')
                    ->options([
                        'success' => 'Verde (Éxito)',
                        'danger' => 'Rojo (Peligro)',
                        'warning' => 'Amarillo (Advertencia)',
                        'info' => 'Azul (Información)',
                        'gray' => 'Gris (Neutral)',
                    ])
                    ->label('Color')
                    ->default('gray')
                    ->required(),
                    
                Forms\Components\TextInput::make('orden')
                    ->numeric()
                    ->default(0)
                    ->label('Orden')
                    ->required(),
                    
                Forms\Components\Toggle::make('activo')
                    ->default(true)
                    ->label('Activo'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('tipo')
                    ->label('Tipo')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->sortable(),
                    
                Tables\Columns\TextColumn::make('color')
                    ->label('Color')
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match($state) {
                        'success' => 'Verde',
                        'danger' => 'Rojo',
                        'warning' => 'Amarillo',
                        'info' => 'Azul',
                        'gray' => 'Gris',
                        default => $state,
                    })
                    ->color(fn (string $state): string => $state),
                    
                Tables\Columns\TextColumn::make('children_count')
                    ->counts('children')
                    ->label('Estados Hijos'),
                    
                Tables\Columns\TextColumn::make('orden')
                    ->label('Orden')
                    ->sortable(),
                    
                Tables\Columns\IconColumn::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->defaultSort('orden', 'asc')
            ->filters([
                Tables\Filters\TernaryFilter::make('activo')
                    ->label('Activo')
                    ->boolean(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListEstadoPadres::route('/'),
            'create' => Pages\CreateEstadoPadre::route('/create'),
            'edit' => Pages\EditEstadoPadre::route('/{record}/edit'),
        ];
    }
}
